/** @file   decorativeobject.cpp
 * @brief   Implementation of DecorativeObject - class.
 * @version $Revision: 1.5 $
 * @author  Tomi Lamminsaari
 */

#include "decorativeobject.h"
#include "www_map.h"
#include "gameanims.h"
#include "soundsamples.h"
#include "settings.h"
#include "redrawqueue.h"
#include "AnimId.h"

using namespace eng2d;
using std::string;

namespace WeWantWar {


/** Constructor
 */
DecorativeObject::DecorativeObject( const string& aTypeName ) :
  GameObject(),
  
 // iObjectType( aType ),
  iAcceptBulletHits( false ),
  iDestroySound( -1 ),
  iAnimationUid( -1 ),
  iObjectTypeName( aTypeName )
{
  this->initObject();
}



/** Destructor
 */
DecorativeObject::~DecorativeObject()
{
}




///
/// Public methods
/// ==============

/** Updates this object.
 */
void DecorativeObject::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      this->setCorrectAnimation( GameAnims::EKilled );
      this->state( GameObject::STATE_KILLED );
    }
    return;
  }
}



/** Redraws us
 */
void DecorativeObject::redraw( RedrawQueue* aQueue )
{
  int sx = MobileObject::m_position.intX() - Map::scrollX;
  int sy = MobileObject::m_position.intY() - Map::scrollY;  
  BITMAP* spriteGfx = m_animation.currentFrame().asBitmap();
  sx -= spriteGfx->w / 2;
  sy -= spriteGfx->h / 2;
  aQueue->addRotatedSprite( RedrawQueue::PRI_BELOW_NORMAL,
                            sx, sy,
                            itofix(this->angle()), spriteGfx );

}



/** Handles the bullet hits
 */
bool DecorativeObject::hitByBullet( Bullet* aBullet )
{
  if ( iAcceptBulletHits == false ) {
    return false;
  }
  // This object can be shot.
  this->causeDamage( aBullet );
  return true;
}



/** Makes sound
 */
void DecorativeObject::makeSound( GameObject::SoundID aSoundId ) const
{
  if ( aSoundId == GameObject::SND_DIE ) {
    if ( iDestroySound != -1 ) {
      Sound::playSample( iDestroySound, false );
    }
  }
}



/** Kills this object
 */
void DecorativeObject::kill()
{
  if ( this->state() == GameObject::STATE_KILLED ||
       this->state() == GameObject::STATE_DYING ) {
    return;
  }
  
  this->setCorrectAnimation( GameAnims::EDying );
  this->state( GameObject::STATE_DYING );
  this->makeSound( GameObject::SND_DIE );
}




///
/// Getter methods
/// ==============

/** Returns the type of this object.
 */
ObjectID::Type DecorativeObject::objectType() const
{
  return ObjectID::TYPE_DECORATIVEOBJECT;
}



/** Tells if we're reloading
 */
bool DecorativeObject::reloading() const
{
  return false;
}



///
/// Private or Protected methods
/// ============================

/** Initizes us.
 */
void DecorativeObject::initObject()
{
  // Construct the whole parameter name and fetch the settings data.
  string paramBounding = iObjectTypeName + "_bsphere:";
  string paramArmor = iObjectTypeName + "_armor:";
  string paramCollision = iObjectTypeName + "_corners:";
  string paramBulletHits = iObjectTypeName + "_bullethits:";
  string paramDestroySound = iObjectTypeName + "_destroysound:";
  string paramAnimUid = iObjectTypeName + "_animUid:";
  int oid = ObjectID::TYPE_DECORATIVEOBJECT;
  if ( paramBounding == "" ) {
    return;
  }
  // Set the attributes
  try {
    this->setArmor( Settings::floatObjProp(oid, paramArmor) );
    this->boundingSphere( Settings::floatObjProp(oid, paramBounding) );
    float coords = Settings::floatObjProp(oid, paramCollision);
    this->setCollisionPoint( 0, Vec2D(-coords,-coords) );
    this->setCollisionPoint( 1, Vec2D( coords,-coords) );
    this->setCollisionPoint( 2, Vec2D(-coords, coords) );
    this->setCollisionPoint( 3, Vec2D( coords, coords) );
    iAcceptBulletHits =
        static_cast<bool>(Settings::intObjProp(oid, paramBulletHits) );
    iDestroySound = Settings::intObjProp(oid, paramDestroySound);
    iAnimationUid = Settings::intObjProp(oid, paramAnimUid);
        
  } catch (...) {
    alert( "DecorativeObject", "initObject()", "Settings failed", "ok",0, 0,0 );
    abort();
  }
  this->setCorrectAnimation( GameAnims::EIdle );
}



/** Sets the correct animation
 */
void DecorativeObject::setCorrectAnimation( int aAnimId )
{
  if ( iAnimationUid != -1 ) {
    const Animation& anim = GameAnims::findAnimation( iAnimationUid, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}


} // end of namespace
